﻿
' ------------------------------------------------------------ Catmull / Cardinal spline
Public Class cSpline

    Private Const nRaw As Int32 = 10
    Private xRaw(nRaw - 1) As Single
    Private yRaw(nRaw - 1) As Single
    Private InitialPoints(nRaw + 1) As PointF
    Private SupportPoints(MAX_BIN_INDEX + 1) As PointF
    Private nDivisions As Int32 = MAX_BIN_INDEX \ (nRaw - 1)
    Private alpha As Single


    ' ===============================================================================================================
    '  Create HEIGHT curve 
    ' ===============================================================================================================
    Friend Sub CreateHeightCurve(ByVal v1 As Single, _
                                 ByVal v2 As Single, _
                                 ByVal v3 As Single, _
                                 ByVal v4 As Single, _
                                 ByVal v5 As Single, _
                                 ByVal v6 As Single, _
                                 ByVal v7 As Single, _
                                 ByVal v8 As Single, _
                                 ByVal e1 As Single, _
                                 ByVal e2 As Single, _
                                 ByVal e3 As Single, _
                                 ByVal e4 As Single, _
                                 ByVal e5 As Single, _
                                 ByVal e6 As Single, _
                                 ByVal e7 As Single, _
                                 ByVal e8 As Single, _
                                 ByVal TkRange As Single, _
                                 ByVal _alpha As Single)

        If _alpha > 1 Then _alpha = 1
        If _alpha < 0 Then _alpha = 0
        alpha = _alpha

        xRaw(0) = 0
        xRaw(1) = e1
        xRaw(2) = e2
        xRaw(3) = e3
        xRaw(4) = e4
        xRaw(5) = e5
        xRaw(6) = e6
        xRaw(7) = e7
        xRaw(8) = e8
        xRaw(9) = 5000

        TkRange *= 100

        yRaw(0) = v1 / TkRange
        yRaw(1) = v1 / TkRange
        yRaw(2) = v2 / TkRange
        yRaw(3) = v3 / TkRange
        yRaw(4) = v4 / TkRange
        yRaw(5) = v5 / TkRange
        yRaw(6) = v6 / TkRange
        yRaw(7) = v7 / TkRange
        yRaw(8) = v8 / TkRange
        yRaw(9) = v8 / TkRange

        If _alpha = 0 Then
            InterpolateHeight_Linear()
        Else
            InitialPoints(0).X = xRaw(0)
            InitialPoints(0).Y = yRaw(0)
            For i As Int32 = 0 To nRaw - 1
                InitialPoints(i + 1).X = xRaw(i)
                InitialPoints(i + 1).Y = yRaw(i)
            Next
            InitialPoints(nRaw + 1).X = xRaw(nRaw - 1)
            InitialPoints(nRaw + 1).Y = yRaw(nRaw - 1)
            CardComputeSupportPoints(InitialPoints)
            InterpolateHeight_CardSpline()
        End If
    End Sub

    ' =================================================== Linear interpolation
    Private Sub InterpolateHeight_Linear()
        For i As Int32 = 0 To MAX_BIN_INDEX
            Spectrometer.HeightLinArray(i) *= CalcAttenuatorCoeff( _
                                                  ComputeLinIntPoint( _
                                                     Spectrometer.BinToEnergy(i)), 300, 1)
        Next
    End Sub

    ' =================================================== Cardinal Spline interpolation
    Private Sub InterpolateHeight_CardSpline()
        For i As Int32 = 0 To MAX_BIN_INDEX
            Spectrometer.HeightLinArray(i) *= CalcAttenuatorCoeff( _
                                                ComputeCardSplineIntPoint( _
                                                    Spectrometer.BinToEnergy(i)), 300, 1)
        Next
    End Sub



    ' ===============================================================================================================
    '  Create ENERGY curve
    ' ===============================================================================================================
    Friend Sub CreateEnergyCurve(ByVal v1 As Single, _
                                 ByVal v2 As Single, _
                                 ByVal v3 As Single, _
                                 ByVal v4 As Single, _
                                 ByVal v5 As Single, _
                                 ByVal v6 As Single, _
                                 ByVal v7 As Single, _
                                 ByVal v8 As Single, _
                                 ByVal e1 As Single, _
                                 ByVal e2 As Single, _
                                 ByVal e3 As Single, _
                                 ByVal e4 As Single, _
                                 ByVal e5 As Single, _
                                 ByVal e6 As Single, _
                                 ByVal e7 As Single, _
                                 ByVal e8 As Single, _
                                 ByVal TkRange As Single, _
                                 ByVal _alpha As Single)

        If _alpha > 1 Then _alpha = 1
        If _alpha < 0 Then _alpha = 0
        alpha = _alpha

        xRaw(0) = 0
        xRaw(1) = e1
        xRaw(2) = e2
        xRaw(3) = e3
        xRaw(4) = e4
        xRaw(5) = e5
        xRaw(6) = e6
        xRaw(7) = e7
        xRaw(8) = e8
        xRaw(9) = 5000

        ' TODO --------------------------------- correct TkRange coefficients
        TkRange *= 100

        yRaw(0) = v1 / TkRange
        yRaw(1) = v1 / TkRange
        yRaw(2) = v2 / TkRange
        yRaw(3) = v3 / TkRange
        yRaw(4) = v4 / TkRange
        yRaw(5) = v5 / TkRange
        yRaw(6) = v6 / TkRange
        yRaw(7) = v7 / TkRange
        yRaw(8) = v8 / TkRange
        yRaw(9) = v8 / TkRange

        If _alpha = 0 Then
            InterpolateEnergy_Linear()
        Else
            InitialPoints(0).X = xRaw(0)
            InitialPoints(0).Y = yRaw(0)
            For i As Int32 = 0 To nRaw - 1
                InitialPoints(i + 1).X = xRaw(i)
                InitialPoints(i + 1).Y = yRaw(i)
            Next
            InitialPoints(nRaw + 1).X = xRaw(nRaw - 1)
            InitialPoints(nRaw + 1).Y = yRaw(nRaw - 1)
            CardComputeSupportPoints(InitialPoints)
            InterpolateEnergy_CardSpline()
        End If
    End Sub

    ' =================================================== Linear interpolation
    Private Sub InterpolateEnergy_Linear()
        For i As Int32 = 0 To MAX_BIN_INDEX
            Spectrometer.EnergyLinArray(i) *= CalcAttenuatorCoeff( _
                                                ComputeLinIntPoint( _
                                                    Spectrometer.BinToEnergy(i)), 300, 1)
        Next
    End Sub

    ' =================================================== Cardinal Spline interpolation
    Private Sub InterpolateEnergy_CardSpline()
        For i As Int32 = 0 To MAX_BIN_INDEX
            Spectrometer.EnergyLinArray(i) *= CalcAttenuatorCoeff( _
                                                ComputeCardSplineIntPoint( _
                                                    Spectrometer.BinToEnergy(i)), 300, 1)
        Next
    End Sub



    ' ===============================================================================================================
    '  LINEAR and SPLINE interpolations
    ' ===============================================================================================================
    Private Function ComputeLinIntPoint(ByVal x As Single) As Single
        Dim i As Int32 = 1
        While i < nRaw - 1 AndAlso xRaw(i) < x
            i = i + 1
        End While
        ComputeLinIntPoint = yRaw(i - 1) + ((x - xRaw(i - 1)) / (xRaw(i) - xRaw(i - 1))) * (yRaw(i) - yRaw(i - 1))
    End Function

    Private Function ComputeCardSplineIntPoint(ByVal x As Single) As Single
        Return SupportPoints(SearchIndex(SupportPoints, x)).Y
    End Function


    ' -----------------------------------------------------------------------------------------------
    '  value = -1 to 1  ( cursor position )
    '  krange = range max of the trimming
    '  kexp = s-curve exponent
    ' -----------------------------------------------------------------------------------------------
    Friend Shared Function CalcAttenuatorCoeff(ByVal value As Single, _
                                               Optional ByVal krange As Single = 1000, _
                                               Optional ByVal kexp As Single = 4) As Single

        value = CSng(Math.Sign(value) * (Math.Abs(value) ^ kexp))

        If value >= 0 Then
            CalcAttenuatorCoeff = value * (krange - 1) + 1
        Else
            CalcAttenuatorCoeff = 1 / ((krange - 1) * -value + 1)
        End If
    End Function

    Private Function SearchIndex(ByVal points As PointF(), _
                                 ByVal n As Single) As Int32
        ' -------------------------------------------------------------------------- Binary search
        Dim iMin As Int32 = 0
        Dim iMax As Int32 = points.Length - 1
        Dim iMid As Int32
        ' -------------------------------------- continue searching while [iMin,iMax] is not empty
        While iMax > iMin
            ' ---------------------------------- calculate the midpoint for roughly equal partition
            iMid = (iMax + iMid) \ 2
            ' ---------------------------------- determine which subarray to search
            If points(iMid).X < n Then
                ' ------------------------------ change min index to search upper subarray
                iMin = iMid + 1
            ElseIf points(iMid).X > n Then
                ' ------------------------------ change max index to search lower subarray
                iMax = iMid - 1
            Else
                ' ------------------------------ searchedX found at index iMid
                Return iMid
            End If
        End While
        ' searchedX not found
        Return iMin
        ' -------------------------------------------------------------------------- Linear search
        'Dim i As Int32 = 0
        'Dim iMax As Int32 = points.Length - 1
        'While i < iMax AndAlso points(i).X < n
        '    i = i + 1
        'End While
        'Return i
    End Function


    ' Computes the support points for the Cardinal Spline curve
    ' Generate spline series of points from array of keyframe points
    ' param "points" array of key frame points
    ' param "numPoints" number of points to generate in spline between each point
    ' returns array of points describing spline
    Private Sub CardComputeSupportPoints(ByVal points As PointF())
        Dim ix As Int32 = 0
        For i As Int32 = 0 To points.Length - 4
            For j As Int32 = 0 To nDivisions - 1
                SupportPoints(ix) = CardPointOnCurve(points(i), _
                                                     points(i + 1), _
                                                     points(i + 2), _
                                                     points(i + 3), _
                                                     (1.0F / nDivisions) * j)
                ix = ix + 1
            Next
        Next
        SupportPoints(ix) = points(points.Length - 2)
    End Sub

    ' Calculates interpolated point between two points using Cardinal Spline
    ' Points calculated exist on the spline between points two and three.
    ' param "p0" First Point
    ' param "p1" Second Point
    ' param "p2" Third Point
    ' param "p3" Fourth Point
    ' param "t"
    ' Normalised distance between second and third point 
    ' where the spline point will be calculated.
    ' Depends on "alfa" tension value between 0.0 and 1.0
    ' Returns Calculated Spline Point
    Private Function CardPointOnCurve(ByVal p0 As PointF, _
                                      ByVal p1 As PointF, _
                                      ByVal p2 As PointF, _
                                      ByVal p3 As PointF, _
                                      ByVal t As Single) As PointF

        Dim t2 As Single = t * t
        Dim t3 As Single = t2 * t

        CardPointOnCurve.X = p1.X + _
        (-alpha * p0.X + alpha * p2.X) * t + _
        (2.0F * alpha * p0.X + (alpha - 3.0F) * p1.X + (3.0F - 2.0F * alpha) * p2.X - alpha * p3.X) * t2 + _
        (-alpha * p0.X + (2.0F - alpha) * p1.X + (alpha - 2.0F) * p2.X + alpha * p3.X) * t3

        CardPointOnCurve.Y = p1.Y + _
        (-alpha * p0.Y + alpha * p2.Y) * t + _
        (2.0F * alpha * p0.Y + (alpha - 3.0F) * p1.Y + (3.0F - 2.0F * alpha) * p2.Y - alpha * p3.Y) * t2 + _
        (-alpha * p0.Y + (2.0F - alpha) * p1.Y + (alpha - 2.0F) * p2.Y + alpha * p3.Y) * t3
    End Function

End Class

